#include <limits>

#include "graph.hpp"

EdgeId const Graph::invalid_edge = std::numeric_limits<EdgeId>::max();
NodeId const Graph::invalid_node = std::numeric_limits<NodeId>::max();

std::ostream & operator<<(std::ostream & os, Graph::Edge const & edge)
{
   os << edge.tail() << " -> " << edge.head() << " Capcacity: " << edge.cap() << " Flow: " << edge.flow() << " Index: " << edge.edge_id();
   return os;
}

std::ostream & operator<<(std::ostream & os, Graph::Node const & node)
{
   os << "Incoming edges: " << node.in_edges().size()
      << " Outgoing edges: " << node.out_edges().size();
   return os;
}


Graph::Graph()
{
}

Graph::Graph(NodeId nodes)
 : nodes(nodes)
{
}

NodeId Graph::add_node() {
   nodes.push_back(Node());
   return nodes.size()-1;
}

void Graph::add_edge(NodeId tail, NodeId head, Capacity cap, EdgeId edge_number)
{
   edges.push_back(Edge(tail, head, cap, edge_number));
   nodes[tail].add_outgoing(edges.size()-1);
   nodes[head].add_incoming(edges.size()-1);
   /*std::cout << "New edge: " << edges.back() << std::endl;*/
}

void Graph::add_edge(Edge const & edge) {
   edges.push_back(edge);
   nodes[edge.tail()].add_outgoing(edges.size()-1);
   nodes[edge.head()].add_incoming(edges.size()-1);
}

void Graph::Node::add_incoming(EdgeId edge)
{
   incoming_.push_back(edge);
}

void Graph::Node::add_outgoing(EdgeId edge)
{
   outgoing_.push_back(edge);
}

Graph::Edge::Edge()
 : tail_(std::numeric_limits<NodeId>::max()),
   head_(std::numeric_limits<NodeId>::max()),
   cap_(std::numeric_limits<Capacity>::max()),
   edge_id_(std::numeric_limits<EdgeId>::max()),
   flow_(std::numeric_limits<Flow>::max())
{
}

Graph::Edge::Edge(NodeId tail, NodeId head, Capacity cap, EdgeId edge_number)
 : tail_(tail),
   head_(head),
   cap_(cap),
   edge_id_(edge_number),
   flow_(std::numeric_limits<Flow>::max())
{
}

NodeId Graph::Edge::tail() const {
   return tail_;
}

NodeId Graph::Edge::head() const {
   return head_;
}

Capacity Graph::Edge::cap() const {
   return cap_;
}

Flow Graph::Edge::flow() const {
   return flow_;
}

void Graph::Edge::set_flow(Flow flow) {
   flow_ = flow;
}

EdgeId Graph::Edge::edge_id() const {
   return edge_id_;
}

std::vector<EdgeId> & Graph::Node::in_edges()
{
   return incoming_;
}

std::vector<EdgeId> & Graph::Node::out_edges()
{
   return outgoing_;
}

std::vector<EdgeId> const & Graph::Node::in_edges() const
{
   return incoming_;
}

std::vector<EdgeId> const & Graph::Node::out_edges() const
{
   return outgoing_;
}

NodeId Graph::num_nodes() const
{
   return nodes.size();
}

EdgeId Graph::num_edges() const
{
   return edges.size();
}

Graph::Node & Graph::node(NodeId node) {
   return nodes[node];
}

Graph::Edge & Graph::edge(EdgeId edge) {
   return edges[edge];
}

