#ifndef GRAPH_HPP
#define GRAPH_HPP

#include <vector>

/** A simple data structure that represents a directed graph */
class Graph
{
public:

   /** Index type for graph nodes. Each node has a unique index from 0 to the
       number of nodes - 1. */
   typedef unsigned int NodeId;

   /** Index type for graph edges. The incoming edges of a node have a unique
       index from 0 to the number of incoming edges - 1. The same holds for
       outgoing edges.
   */
   typedef unsigned int EdgeId;

   /** Create a graph with a given number of nodes */
   Graph(NodeId nodes);

   /** Add a directed edge from the node tail to the node head. */
   void add_edge(NodeId tail_idx, NodeId head_idx);

   /** Return the number of nodes of this graph. */
   NodeId num_nodes() const;

   /** Return the number of edges leaving the given node. */
   EdgeId num_outgoing_edges(NodeId node_idx) const;

   /** Return the number of edges entering the given node. */
   EdgeId num_incoming_edges(NodeId node_idx) const;

   /** Return the head of edge with index edge_idx leaving the given node. */
   NodeId outgoing_edge_head(NodeId node_idx, EdgeId edge_idx) const;

   /** Return the tail of edge with index edge_idx entering the given node. */
   NodeId incoming_edge_tail(NodeId node_idx, EdgeId edge_idx) const;

private:

   /** Information about a node of the graph. */
   struct Node
   {
      /** Add an incoming edge from node tail */
      void add_incoming(NodeId tail);

      /** Add an outgoing edge to node head */
      void add_outgoing(NodeId head);

      /** Source and targets of the edges entering and leaving this node. */
      std::vector<NodeId> incoming;
      std::vector<NodeId> outgoing;
   };

   std::vector<Node> nodes;
};

#endif
